<?php

Yii::import('application.modules.sms.models.base.BaseStudentIdealList');

/**
 * Model class for StudentIdealLists
 *
 *
 * @author Shiva <shiva.dharana@univariety.com>
 * @version 1.0
 * @package sms.models
 *
 */
class StudentIdealList extends BaseStudentIdealList{

    /**
     * Statuses of the student list
     * @var const
     */
    const STATUS_STARTED = 'Started';
    const STATUS_UNIDONE = 'Unidone';
    const STATUS_CONFIRMED = 'Confirmed';
    const STATUS_INVALID = 'Invalid';

    const TYPE_ASPIRATIONAL = 'Aspirational';
    const TYPE_IDEAL = 'Ideal';
    const TYPE_SAFE = 'Safe';
    const TYPE_BEST_FIT = 'BestFit';

    public static function model($className=__CLASS__){
        return parent::model($className);
    }

    public static function getStatuses(){
        return array(
        self::STATUS_STARTED,
        self::STATUS_UNIDONE,
        self::STATUS_CONFIRMED,
        self::STATUS_INVALID,
        );
    }
    
	public static function getTypeLabels(){
        return array(
            self::TYPE_ASPIRATIONAL => 'Aspiration',
	        self::TYPE_SAFE => 'Safe',
	        self::TYPE_BEST_FIT => 'Best Fit',
        );
    }

    /**
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        // NOTE: you should only define rules for those attributes that
        // will receive user inputs.
        return array(
        array('*', 'validateCollegeCount', 'on' => 'confirm'),
        array('user_ref_id, department_ref_id', 'required'),
        array('department_ref_id', 'numerical', 'integerOnly'=>true),
        array('user_ref_id', 'length', 'max'=>10),
        array('type', 'length', 'max'=>12),
        array('status', 'length', 'max'=>9),
        array('conclude_date', 'safe'),
        // The following rule is used by search().
        // Please remove those attributes that should not be searched.
        array('student_ideal_list_id, user_ref_id, department_ref_id, type, status, added_on, conclude_date', 'safe', 'on'=>'search'),
        );
    }

    /**
     * @return array relational rules.
     */
    public function relations()
    {
        // NOTE: you may need to adjust the relation name and the related
        // class name for the relations automatically generated below.
        return array(
			'student' => array(self::BELONGS_TO, 'TblUserMaster', 'user_ref_id'),
			'departmentRef' => array(self::BELONGS_TO, 'LookupDepartment', 'department_ref_id'),
			'collegeList' => array(self::HAS_MANY, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "status != '" . StudentIdealListCollege::STATUS_REJECTED . "'", 'order' => 'preference_order'),
            'acceptedCount' => array(self::STAT, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "status = '" . StudentIdealListCollege::STATUS_ACCEPTED . "'"),
            'pendingCount' => array(self::STAT, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "status = '" . StudentIdealListCollege::STATUS_PENDING . "'"),
            'reviewedCount' => array(self::STAT, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "status = '" . StudentIdealListCollege::STATUS_REVIEWED . "'"),
            'studentAddedCount' => array(self::STAT, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "suggest_type = '" . StudentIdealListCollege::SUGGEST_TYPE_STUDENT . "' AND status != '" . StudentIdealListCollege::STATUS_REJECTED . "'"),
        	'advisorAddedCount' => array(self::STAT, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "suggest_type = '" . StudentIdealListCollege::SUGGEST_TYPE_ADVISOR . "'"),
        	'spocAddedCount' => array(self::STAT, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "suggest_type = '" . StudentIdealListCollege::SUGGEST_TYPE_SPOC . "'"),
            'acceptedCollegeList' => array(self::HAS_MANY, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "status = '" . StudentIdealListCollege::STATUS_ACCEPTED . "'", 'order' => 'preference_order'),
            'applyingCollegeList' => array(self::HAS_MANY, 'StudentIdealListCollege', 'student_ideal_list_ref_id', 'condition' => "is_applying = '" . StudentIdealListCollege::APPLYING . "' AND status = '" . StudentIdealListCollege::STATUS_ACCEPTED . "'", 'order' => 'preference_order'),
        );
    }

    /**
     * This is invoked before the record is saved.
     * @return boolean whether the record should be saved.
     */
    protected function beforeSave()
    {
        if(parent::beforeSave()){
            if($this->isNewRecord){
                $this->added_on = date('Y-m-d H:i:s');
                $this->status = self::STATUS_STARTED;
            }else{
                $this->conclude_date = date('Y-m-d H:i:s');
            }
            return true;
        }
        return false;
    }

    /**
     * Either creates a list based on studentId, departmentId, and type if not exists,
     * else returns the existing record
     * @param int $studentId
     * @param int $departmentId
     * @param int $type
     * @return StudentIdealList stdObj
     */
    public static function getRecord($studentId, $departmentId, $type, $autoInsert = true){
        // Check for existing active records
        $criteria = new CDbCriteria();
        $criteria->condition = "status != '" . self::STATUS_INVALID . "'";
        $criteria->compare('user_ref_id', $studentId);
        $criteria->compare('type', $type);
        $criteria->compare('department_ref_id', $departmentId);

        if(self::model()->exists($criteria)){
            return self::model()->find($criteria);
        }
        if($autoInsert){
            $studentIdealList = new StudentIdealList;
            $studentIdealList->user_ref_id = $studentId;
            $studentIdealList->type = $type;
            $studentIdealList->department_ref_id = $departmentId;
            $studentIdealList->save();
            return $studentIdealList;
        }
    }

    public function validateCollegeCount(){
        if($this->acceptedCount < 1){
            $this->addError('*', 'Please accept atleast one college');
            return false;
        }
        if($this->acceptedCount > 8){
            $this->addError('*', 'Please accept max 8 colleges');
            return false;
        }
        return false;
    }
    
    /**
     * Returns bool to indicate whether the list is ready to confirm or not.
     * @return boolean
     */
    public function isReadyToConfirm(){
        return ($this->status == self::STATUS_UNIDONE && $this->pendingCount == 0 && $this->reviewedCount == 0 && $this->acceptedCount > 0);
    }
    
    /**
     * Checks whether the safe list for given department is confirmed or not
     * @param TblUserMaster $student
     * @param int $departmentId
     * @return boolean
     */
    public static function isSafeListConfirmed($student, $departmentId){
        $list = self::getRecord($student->user_id, $departmentId, self::TYPE_SAFE);
        return $list->status == self::STATUS_CONFIRMED;
    }

    public static function addStudentIdealListAndCollege($user_ref_id=0, $deptId=0, $universityId=0, $listType = StudentIdealList::TYPE_ASPIRATIONAL){
        $studentIdealList = StudentIdealList::getRecord($user_ref_id, $deptId, $listType);
        //$studentIdealList = StudentIdealList::getRecord($fields['user_ref_id'], $deptArray['department_id'], StudentIdealList::TYPE_ASPIRATIONAL);

        //creating student_ideal_list_country record
        $studentIdealListCollege = new StudentIdealListCollege;
        $studentIdealListCollege->uni_ref_id = $universityId;
        $studentIdealListCollege->suggest_type = StudentIdealListCollege::SUGGEST_TYPE_STUDENT;
        $bl = $studentIdealListCollege->addUniversity($studentIdealList);
        //echo "<br>UserID:".$fields['user_ref_id']."-DeptId:".$deptArray['main_dept_id'];
        if($bl){
            return true;
        }else{
            //var_dump($studentIdealListCollege->getErrors());
            return false;
        }
    }
}